const { body, param, query } = require('express-validator');
const { validateRequest } = require('./validationMiddleware');

const validateCreateSupplier = [
  body('name')
    .trim()
    .notEmpty()
    .withMessage('Supplier name is required')
    .isLength({ max: 100 })
    .withMessage('Supplier name must be at most 100 characters'),

  body('supplierType')
    .notEmpty()
    .withMessage('Supplier type is required')
    .isIn(['GOODS', 'SERVICES', 'BOTH'])
    .withMessage('Invalid supplier type'),

  body('contactPerson')
    .optional()
    .trim()
    .isLength({ max: 100 })
    .withMessage('Contact person name must be at most 100 characters'),

  body('phone')
    .optional()
    .trim()
    .isLength({ max: 20 })
    .withMessage('Phone number must be at most 20 characters'),

  body('email')
    .optional()
    .trim()
    .isEmail()
    .withMessage('Invalid email format')
    .isLength({ max: 100 })
    .withMessage('Email must be at most 100 characters'),

  body('address')
    .optional()
    .trim()
    .isLength({ max: 255 })
    .withMessage('Address must be at most 255 characters'),

  body('taxId')
    .optional()
    .trim()
    .isLength({ max: 50 })
    .withMessage('Tax ID must be at most 50 characters'),

  validateRequest
];

const validateUpdateSupplier = [
  param('id')
    .isInt()
    .withMessage('Invalid supplier ID'),

  ...validateCreateSupplier,

  body('isActive')
    .optional()
    .isBoolean()
    .withMessage('isActive must be a boolean value'),

  validateRequest
];

const validateUpdateAccount = [
  param('supplierId')
    .isInt()
    .withMessage('Invalid supplier ID'),

  body('totalPurchases')
    .optional()
    .isDecimal({ decimal_digits: '0,2' })
    .withMessage('Total purchases must be a valid decimal with up to 2 decimal places')
    .custom((value) => {
      if (parseFloat(value) < 0) {
        throw new Error('Total purchases cannot be negative');
      }
      return true;
    }),

  body('balanceDue')
    .optional()
    .isDecimal({ decimal_digits: '0,2' })
    .withMessage('Balance due must be a valid decimal with up to 2 decimal places')
    .custom((value) => {
      if (parseFloat(value) < 0) {
        throw new Error('Balance due cannot be negative');
      }
      return true;
    }),

  validateRequest
];

const validateSupplierQuery = [
  query('type')
    .optional()
    .isIn(['GOODS', 'SERVICES', 'BOTH'])
    .withMessage('Invalid supplier type'),

  query('isActive')
    .optional()
    .isBoolean()
    .withMessage('isActive must be a boolean value'),

  query('search')
    .optional()
    .trim()
    .isLength({ max: 100 })
    .withMessage('Search term must be at most 100 characters'),

  validateRequest
];

module.exports = {
  validateCreateSupplier,
  validateUpdateSupplier,
  validateUpdateAccount,
  validateSupplierQuery
};
