const { body, param, query } = require('express-validator');
const { validateRequest } = require('./validationMiddleware');

const validateCreateItem = [
  body('code')
    .trim()
    .notEmpty()
    .withMessage('Item code is required')
    .isLength({ max: 50 })
    .withMessage('Item code must be at most 50 characters')
    .matches(/^[A-Za-z0-9-_]+$/)
    .withMessage('Item code can only contain letters, numbers, hyphens, and underscores'),

  body('name')
    .trim()
    .notEmpty()
    .withMessage('Item name is required')
    .isLength({ max: 100 })
    .withMessage('Item name must be at most 100 characters'),

  body('category')
    .trim()
    .notEmpty()
    .withMessage('Category is required')
    .isLength({ max: 100 })
    .withMessage('Category must be at most 100 characters'),

  body('unitPrice')
    .notEmpty()
    .withMessage('Unit price is required')
    .isFloat({ min: 0 })
    .withMessage('Unit price must be a positive number'),

  body('stock')
    .optional()
    .isInt({ min: 0 })
    .withMessage('Stock must be a non-negative integer'),

  body('reorderLevel')
    .optional()
    .isInt({ min: 0 })
    .withMessage('Reorder level must be a non-negative integer'),

  body('description')
    .optional()
    .trim(),

  validateRequest
];

const validateUpdateItem = [
  param('id')
    .isInt()
    .withMessage('Invalid item ID'),

  ...validateCreateItem
];

const validateCreateLocation = [
  body('locationName')
    .trim()
    .notEmpty()
    .withMessage('Location name is required')
    .isLength({ max: 100 })
    .withMessage('Location name must be at most 100 characters'),

  body('description')
    .optional()
    .trim(),

  validateRequest
];

const validateUpdateLocation = [
  param('id')
    .isInt()
    .withMessage('Invalid location ID'),

  body('locationName')
    .trim()
    .notEmpty()
    .withMessage('Location name is required')
    .isLength({ max: 100 })
    .withMessage('Location name must be at most 100 characters'),

  body('description')
    .optional()
    .trim(),

  body('isActive')
    .optional()
    .isBoolean()
    .withMessage('isActive must be a boolean value'),

  validateRequest
];

const validateUpdateStock = [
  param('itemId')
    .isInt()
    .withMessage('Invalid item ID'),

  param('locationId')
    .isInt()
    .withMessage('Invalid location ID'),

  body('quantity')
    .isInt({ min: 0 })
    .withMessage('Quantity must be a non-negative integer'),

  body('minimumQuantity')
    .isInt({ min: 0 })
    .withMessage('Minimum quantity must be a non-negative integer')
    .custom((value, { req }) => {
      if (parseInt(value) > parseInt(req.body.maximumQuantity)) {
        throw new Error('Minimum quantity cannot be greater than maximum quantity');
      }
      return true;
    }),

  body('maximumQuantity')
    .isInt({ min: 0 })
    .withMessage('Maximum quantity must be a non-negative integer'),

  validateRequest
];

const validateInventoryQuery = [
  query('search')
    .optional()
    .trim()
    .isLength({ max: 100 })
    .withMessage('Search term must be at most 100 characters'),

  query('isActive')
    .optional()
    .isBoolean()
    .withMessage('isActive must be a boolean value'),

  validateRequest
];

module.exports = {
  validateCreateItem,
  validateUpdateItem,
  validateCreateLocation,
  validateUpdateLocation,
  validateUpdateStock,
  validateInventoryQuery
};
