const { body, param, query } = require('express-validator');
const { validateRequest } = require('./validationMiddleware');

const validateInitializeCapital = [
  body('locationId')
    .notEmpty()
    .withMessage('Location ID is required')
    .isInt()
    .withMessage('Invalid location ID'),
  
  body('openingBalance')
    .notEmpty()
    .withMessage('Opening balance is required')
    .isDecimal({ decimal_digits: '0,2' })
    .withMessage('Opening balance must be a valid decimal with up to 2 decimal places')
    .custom((value) => {
      if (parseFloat(value) < 0) {
        throw new Error('Opening balance cannot be negative');
      }
      return true;
    }),
  
  validateRequest
];

const validateTransaction = [
  body('capitalId')
    .notEmpty()
    .withMessage('Capital account ID is required')
    .isInt()
    .withMessage('Invalid capital account ID'),
  
  body('transactionType')
    .notEmpty()
    .withMessage('Transaction type is required')
    .isIn(['INJECTION', 'DRAWING'])
    .withMessage('Transaction type must be either INJECTION or DRAWING'),
  
  body('amount')
    .notEmpty()
    .withMessage('Amount is required')
    .isDecimal({ decimal_digits: '0,2' })
    .withMessage('Amount must be a valid decimal with up to 2 decimal places')
    .custom((value) => {
      if (parseFloat(value) <= 0) {
        throw new Error('Amount must be greater than 0');
      }
      return true;
    }),
  
  body('description')
    .optional()
    .trim()
    .isLength({ max: 255 })
    .withMessage('Description must be at most 255 characters'),
  
  validateRequest
];

const validateDateRange = [
  query('startDate')
    .optional()
    .isISO8601()
    .withMessage('Start date must be a valid ISO 8601 date'),
  
  query('endDate')
    .optional()
    .isISO8601()
    .withMessage('End date must be a valid ISO 8601 date')
    .custom((endDate, { req }) => {
      if (endDate && req.query.startDate && new Date(endDate) < new Date(req.query.startDate)) {
        throw new Error('End date must be after start date');
      }
      return true;
    }),
  
  validateRequest
];

module.exports = {
  validateInitializeCapital,
  validateTransaction,
  validateDateRange
};
