const prisma = require('../config/database');
const { hashPassword } = require('../utils/password');

const register = async (req, res) => {
  try {
    const { username, password, fullName, email, phone, locationId } = req.body;

    // Check if location exists
    const location = await prisma.location.findUnique({
      where: { id: parseInt(locationId) }
    });

    if (!location) {
      return res.status(400).json({ error: 'Invalid location selected' });
    }

    const hashedPassword = await hashPassword(password);

    const user = await prisma.user.create({
      data: {
        username,
        passwordHash: hashedPassword,
        fullName,
        email,
        phone,
        locationId: parseInt(locationId),
        userRole: 'CASHIER', // Default role for new registrations
        isActive: true,
        verificationStatus: 'PENDING'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        locationId: true,
        verificationStatus: true,
        createdAt: true
      }
    });

    res.status(201).json({
      message: 'Registration successful. Please wait for admin verification.',
      user
    });
  } catch (error) {
    if (error.code === 'P2002') {
      return res.status(400).json({ 
        error: 'Username or email already exists' 
      });
    }
    res.status(500).json({ error: error.message });
  }
};

const getPendingVerifications = async (req, res) => {
  try {
    const pendingUsers = await prisma.user.findMany({
      where: {
        verificationStatus: 'PENDING'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        locationId: true,
        createdAt: true,
        location: {
          select: {
            name: true,
            type: true
          }
        }
      },
      orderBy: {
        createdAt: 'desc'
      }
    });

    res.json(pendingUsers);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const verifyUser = async (req, res) => {
  try {
    const { userId } = req.params;
    const { status, note } = req.body;

    if (!['APPROVED', 'REJECTED'].includes(status)) {
      return res.status(400).json({ error: 'Invalid verification status' });
    }

    const user = await prisma.user.update({
      where: { id: parseInt(userId) },
      data: {
        verificationStatus: status,
        verificationNote: note,
        verifiedBy: req.admin.id,
        verifiedAt: new Date(),
        isActive: status === 'APPROVED'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        verificationStatus: true,
        verificationNote: true,
        verifiedAt: true,
        location: {
          select: {
            name: true
          }
        }
      }
    });

    res.json({
      message: `User ${status.toLowerCase()} successfully`,
      user
    });
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'User not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getUsersByLocation = async (req, res) => {
  try {
    const { locationId } = req.params;
    
    const users = await prisma.user.findMany({
      where: {
        locationId: parseInt(locationId),
        verificationStatus: 'APPROVED'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        userRole: true,
        isActive: true,
        lastLogin: true
      },
      orderBy: {
        fullName: 'asc'
      }
    });

    res.json(users);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

/**
 * Get all users in the database
 * Admin only endpoint
 */
const getAllUsers = async (req, res) => {
  try {
    const users = await prisma.user.findMany({
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        userRole: true,
        locationId: true,
        isActive: true,
        verificationStatus: true,
        lastLogin: true,
        createdAt: true,
        location: {
          select: {
            id: true,
            name: true,
            type: true
          }
        }
      },
      orderBy: {
        fullName: 'asc'
      }
    });

    res.json(users);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

/**
 * Get a user by ID
 * Admin only endpoint
 */
const getUserById = async (req, res) => {
  try {
    const { userId } = req.params;
    
    const user = await prisma.user.findUnique({
      where: { id: parseInt(userId) },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        userRole: true,
        locationId: true,
        isActive: true,
        verificationStatus: true,
        lastLogin: true,
        createdAt: true,
        verifiedBy: true,
        verifiedAt: true,
        verificationNote: true,
        location: {
          select: {
            id: true,
            name: true,
            type: true
          }
        }
      }
    });

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json(user);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  register,
  getPendingVerifications,
  verifyUser,
  getUsersByLocation,
  getAllUsers,
  getUserById
};
