const prisma = require('../config/database');

const createLocation = async (req, res) => {
  try {
    const { name, type, address, contactPhone, contactEmail } = req.body;

    const location = await prisma.location.create({
      data: {
        name,
        type,
        address,
        contactPhone,
        contactEmail,
        isActive: true
      }
    });

    res.status(201).json(location);
  } catch (error) {
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Location name must be unique' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getLocations = async (req, res) => {
  try {
    const locations = await prisma.location.findMany({
      orderBy: { name: 'asc' }
    });
    res.json(locations);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getLocationById = async (req, res) => {
  try {
    const { id } = req.params;
    const location = await prisma.location.findUnique({
      where: { id: parseInt(id) },
      include: {
        users: {
          select: {
            id: true,
            username: true,
            fullName: true,
            userRole: true,
            isActive: true
          }
        }
      }
    });

    if (!location) {
      return res.status(404).json({ error: 'Location not found' });
    }

    res.json(location);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateLocation = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, type, address, contactPhone, contactEmail, isActive } = req.body;

    const location = await prisma.location.update({
      where: { id: parseInt(id) },
      data: {
        name,
        type,
        address,
        contactPhone,
        contactEmail,
        isActive: isActive ?? undefined
      }
    });

    res.json(location);
  } catch (error) {
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Location name must be unique' });
    }
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Location not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

const deleteLocation = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Check if location has associated users
    const locationWithUsers = await prisma.location.findUnique({
      where: { id: parseInt(id) },
      include: { users: true }
    });

    if (locationWithUsers?.users.length > 0) {
      return res.status(400).json({ 
        error: 'Cannot delete location with associated users. Please reassign or remove users first.' 
      });
    }

    await prisma.location.delete({
      where: { id: parseInt(id) }
    });

    res.status(204).send();
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Location not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  createLocation,
  getLocations,
  getLocationById,
  updateLocation,
  deleteLocation
};
