const prisma = require('../config/database');

// Category Management
const createCategory = async (req, res) => {
  try {
    const { name, description } = req.body;

    const category = await prisma.inventoryCategory.create({
      data: {
        name,
        description
      }
    });

    res.status(201).json(category);
  } catch (error) {
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Category name already exists' });
    }
    res.status(500).json({ error: error.message });
  }
};

const updateCategory = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, description } = req.body;

    const category = await prisma.inventoryCategory.update({
      where: { id: parseInt(id) },
      data: {
        name,
        description
      }
    });

    res.json(category);
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Category not found' });
    }
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Category name already exists' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getCategory = async (req, res) => {
  try {
    const { id } = req.params;

    const category = await prisma.inventoryCategory.findUnique({
      where: { id: parseInt(id) },
      include: {
        items: true
      }
    });

    if (!category) {
      return res.status(404).json({ error: 'Category not found' });
    }

    // Add item count to the response
    const formattedCategory = {
      ...category,
      itemCount: category.items.length
    };

    res.json(formattedCategory);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getAllCategories = async (req, res) => {
  try {
    const categories = await prisma.inventoryCategory.findMany({
      include: {
        items: true
      },
      orderBy: {
        name: 'asc'
      }
    });

    // Format categories to match frontend expectations
    const formattedCategories = categories.map(category => ({
      id: category.id,
      name: category.name,
      description: category.description || '',
      itemCount: category.items.length
    }));

    res.json(formattedCategories);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  createCategory,
  updateCategory,
  getCategory,
  getAllCategories
};
