const prisma = require('../config/database');

const initializeCapitalAccount = async (req, res) => {
  try {
    const { locationId, openingBalance } = req.body;

    // Check if capital account already exists for this location
    const existingAccount = await prisma.capitalAccount.findFirst({
      where: { locationId: parseInt(locationId) }
    });

    if (existingAccount) {
      return res.status(400).json({ 
        error: 'Capital account already exists for this location' 
      });
    }

    const capitalAccount = await prisma.capitalAccount.create({
      data: {
        locationId: parseInt(locationId),
        openingBalance: openingBalance,
        currentBalance: openingBalance
      }
    });

    res.status(201).json(capitalAccount);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getLocationCapital = async (req, res) => {
  try {
    const { locationId } = req.params;

    const capitalAccount = await prisma.capitalAccount.findFirst({
      where: { locationId: parseInt(locationId) },
      include: {
        location: {
          select: {
            name: true,
            type: true
          }
        }
      }
    });

    if (!capitalAccount) {
      return res.status(404).json({ error: 'Capital account not found' });
    }

    res.json(capitalAccount);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const recordTransaction = async (req, res) => {
  const { capitalId, transactionType, amount, description } = req.body;

  try {
    // Start transaction
    const result = await prisma.$transaction(async (prisma) => {
      // Get current capital account
      const capitalAccount = await prisma.capitalAccount.findUnique({
        where: { id: parseInt(capitalId) }
      });

      if (!capitalAccount) {
        throw new Error('Capital account not found');
      }

      // Calculate new balance
      let newBalance = capitalAccount.currentBalance;
      if (transactionType === 'INJECTION') {
        newBalance = newBalance.add(amount);
      } else if (transactionType === 'DRAWING') {
        if (capitalAccount.currentBalance.lessThan(amount)) {
          throw new Error('Insufficient balance for drawing');
        }
        newBalance = newBalance.sub(amount);
      }

      // Create transaction record
      const transaction = await prisma.capitalTransaction.create({
        data: {
          capitalId: parseInt(capitalId),
          transactionType,
          amount,
          description,
          recordedById: req.user.id
        }
      });

      // Update capital account balance
      const updatedAccount = await prisma.capitalAccount.update({
        where: { id: parseInt(capitalId) },
        data: {
          currentBalance: newBalance,
          lastTransactionDate: new Date()
        }
      });

      return { transaction, updatedAccount };
    });

    res.status(201).json(result);
  } catch (error) {
    if (error.message === 'Insufficient balance for drawing') {
      return res.status(400).json({ error: error.message });
    }
    res.status(500).json({ error: error.message });
  }
};

const getTransactionHistory = async (req, res) => {
  try {
    const { capitalId } = req.params;
    const { startDate, endDate, type } = req.query;

    let whereClause = {
      capitalId: parseInt(capitalId)
    };

    if (startDate && endDate) {
      whereClause.transactionDate = {
        gte: new Date(startDate),
        lte: new Date(endDate)
      };
    }

    if (type && ['INJECTION', 'DRAWING'].includes(type)) {
      whereClause.transactionType = type;
    }

    const transactions = await prisma.capitalTransaction.findMany({
      where: whereClause,
      include: {
        recordedBy: {
          select: {
            username: true,
            fullName: true
          }
        }
      },
      orderBy: {
        transactionDate: 'desc'
      }
    });

    res.json(transactions);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getCapitalSummary = async (req, res) => {
  try {
    const { locationId } = req.params;
    const { startDate, endDate } = req.query;

    const capitalAccount = await prisma.capitalAccount.findFirst({
      where: { locationId: parseInt(locationId) },
      include: {
        transactions: {
          where: {
            transactionDate: {
              gte: startDate ? new Date(startDate) : undefined,
              lte: endDate ? new Date(endDate) : undefined
            }
          }
        }
      }
    });

    if (!capitalAccount) {
      return res.status(404).json({ error: 'Capital account not found' });
    }

    // Calculate summary
    const summary = {
      openingBalance: capitalAccount.openingBalance,
      currentBalance: capitalAccount.currentBalance,
      totalInjections: capitalAccount.transactions
        .filter(t => t.transactionType === 'INJECTION')
        .reduce((sum, t) => sum.add(t.amount), new Decimal(0)),
      totalDrawings: capitalAccount.transactions
        .filter(t => t.transactionType === 'DRAWING')
        .reduce((sum, t) => sum.add(t.amount), new Decimal(0)),
      transactionCount: capitalAccount.transactions.length,
      lastTransaction: capitalAccount.lastTransactionDate
    };

    res.json(summary);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  initializeCapitalAccount,
  getLocationCapital,
  recordTransaction,
  getTransactionHistory,
  getCapitalSummary
};
