const prisma = require('../config/database');
const { hashPassword, comparePassword } = require('../utils/password');
const { generateToken } = require('../utils/jwt');

const login = async (req, res) => {
  try {
    const { username, password } = req.body;

    const user = await prisma.user.findUnique({
      where: { username },
      select: {
        id: true,
        username: true,
        passwordHash: true,
        userRole: true,
        isActive: true,
        locationId: true,
        location: {
          select: {
            id: true,
            name: true,
            type: true
          }
        }
      }
    });

    if (!user || !user.isActive) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const isValidPassword = await comparePassword(password, user.passwordHash);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Update last login
    await prisma.user.update({
      where: { id: user.id },
      data: { lastLogin: new Date() }
    });

    const token = generateToken({
      userId: user.id,
      username: user.username,
      role: user.userRole
    });

    res.json({
      token,
      user: {
        id: user.id,
        username: user.username,
        role: user.userRole,
        locationId: user.locationId,
        locationType: user.location?.type || null,
        locationName: user.location?.name || null
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getProfile = async (req, res) => {
  try {
    const user = await prisma.user.findUnique({
      where: { id: req.user.id },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        userRole: true,
        locationId: true,
        isActive: true,
        lastLogin: true,
        location: {
          select: {
            id: true,
            name: true,
            type: true
          }
        }
      }
    });

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json(user);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

/**
 * Register a new user with PENDING verification status
 */
const register = async (req, res) => {
  try {
    const { username, password, fullName, email, phone } = req.body;

    // Hash the password
    const passwordHash = await hashPassword(password);

    // Create the user with PENDING verification status
    const newUser = await prisma.user.create({
      data: {
        username,
        passwordHash,
        fullName,
        email,
        phone,
        userRole: 'CASHIER', // Default role, will be updated by admin during approval
        verificationStatus: 'PENDING'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        verificationStatus: true,
        createdAt: true
      }
    });

    res.status(201).json({
      message: 'Registration successful. Your account is pending approval by an administrator.',
      user: newUser
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

/**
 * Get all pending users (for admin approval)
 */
const getPendingUsers = async (req, res) => {
  try {
    // Since we're making this endpoint public for now, we'll skip the admin check
    // In a production environment, you should re-add the authenticate and authorizeAdmin middleware
    
    const pendingUsers = await prisma.user.findMany({
      where: { verificationStatus: 'PENDING' },
      select: {
        id: true,
        username: true,
        fullName: true,
        email: true,
        phone: true,
        createdAt: true
      },
      orderBy: { createdAt: 'desc' }
    });

    res.json(pendingUsers);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

/**
 * Approve or reject a user registration
 */
const approveUser = async (req, res) => {
  try {

    const { userId, locationId, userRole, verificationStatus, verificationNote } = req.body;

    // Check if user exists and is pending
    const user = await prisma.user.findUnique({
      where: { id: userId },
      select: { id: true, verificationStatus: true }
    });

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    if (user.verificationStatus !== 'PENDING') {
      return res.status(400).json({ error: 'User is not in pending status' });
    }

    // Update user with approval/rejection details
    const updatedUser = await prisma.user.update({
      where: { id: userId },
      data: {
        locationId: verificationStatus === 'APPROVED' ? locationId : null,
        userRole: verificationStatus === 'APPROVED' ? userRole : 'CASHIER',
        verificationStatus,
        verificationNote,
        verifiedBy: req.user.id,
        verifiedAt: new Date(),
        isActive: verificationStatus === 'APPROVED'
      },
      select: {
        id: true,
        username: true,
        fullName: true,
        userRole: true,
        locationId: true,
        verificationStatus: true,
        verificationNote: true,
        verifiedAt: true,
        location: {
          select: {
            name: true,
            type: true
          }
        }
      }
    });

    res.json({
      message: `User ${verificationStatus === 'APPROVED' ? 'approved' : 'rejected'} successfully`,
      user: updatedUser
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  login,
  getProfile,
  register,
  getPendingUsers,
  approveUser
};
